/* main.c - eTimer Blink example for S32R274 */
/* Description:  Blinks LED based on eTimer */
/* Rev 1.0 Aug 13 2018 D Chung - production version */
/* Copyright NXP Semiconductor, Inc 2018 All rights reserved. */

/*******************************************************************************
* NXP Semiconductor Inc.
* (c) Copyright 2018 NXP Semiconductor, Inc.
* ALL RIGHTS RESERVED.
********************************************************************************
Services performed by NXP in this matter are performed AS IS and without
any warranty. CUSTOMER retains the final decision relative to the total design
and functionality of the end product. NXP neither guarantees nor will be
held liable by CUSTOMER for the success of this project.
NXP DISCLAIMS ALL WARRANTIES, EXPRESSED, IMPLIED OR STATUTORY INCLUDING,
BUT NOT LIMITED TO, IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR
A PARTICULAR PURPOSE ON ANY HARDWARE, SOFTWARE ORE ADVISE SUPPLIED TO THE PROJECT
BY NXP, AND OR NAY PRODUCT RESULTING FROM NXP SERVICES. IN NO EVENT
SHALL NXP BE LIABLE FOR INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF
THIS AGREEMENT.

CUSTOMER agrees to hold NXP harmless against any and all claims demands or
actions by anyone on account of any damage, or injury, whether commercial,
contractual, or tortuous, rising directly or indirectly as a result of the advise
or assistance supplied CUSTOMER in connection with product, services or goods
supplied under this Agreement.

Copyright 2018 NXP.  This software is owned or controlled by NXP and may only
be used strictly in accordance with the applicable license terms.  By expressly
accepting such terms or by downloading, installing, activating and/or otherwise
using the software, you are agreeing that you have read, and that you agree to
comply with and are bound by, such license terms.  If you do not agree to be
bound by the applicable license terms, then you may not retain, install, activate
or otherwise use the software.
********************************************************************************
* File:             main_Z4.c
* Owner:            David Chung
* Version:          1.0
* Date:             Aug-13-2018
* Classification:   General Business Information
* Brief:            ETimer count mode demonstration
********************************************************************************
* Detailed Description:
* Application performs basic initialization, setup PLL1 to 200 MHz.
* PLL1 is system frequency, PLL0 in initialized to 50MHz
* initializes peripherals clock (MC_CLK is set to 5MHz)
* initializes eTimer to count mode
* initializes interrupts, blinking one LED by eTimer interrupt
* once per second
*
* ------------------------------------------------------------------------------
* Test HW:         S32R274RRUEVB+MPC57xxMB
* MCU:             S32R274
* Fsys:            200 MHz
* Debugger:        USB Multilink
* Target:          FLASH
* EVB connection:  Motherboard connections-
* 					Connect PA0 (P8.1) to user LED (P7.1)
*
* ------------------------------------------------------------------------------
*
********************************************************************************
Revision History:
Version  Date            Author  			Description of Changes
1.0      Aug-13-2018     David Chung  		Initial Version
*******************************************************************************/

#include "derivative.h" /* include peripheral declarations */

/*******************************************************************************
* Constants and macros
*******************************************************************************/
#define LED1_pin 0
#define KEY_VALUE1 0x5AF0ul
#define KEY_VALUE2 0xA50Ful

/*******************************************************************************
* Function prototypes
*******************************************************************************/
static void SysClk_Init(void);
static void InitPeriClkGen(void);
static void GPIO_Init(void);
static void HW_Init(void);
static void ETimer_Init(void);
void ETimer_ISR(void);

extern void xcptn_xmpl(void);

void hw_init(void)
{
#if defined(TURN_ON_CPU1) || defined(TURN_ON_CPU2)
	uint32_t mctl = MC_ME.MCTL.R;
#endif
#if defined(TURN_ON_CPU1)
	/* enable core 1 in all modes */
	MC_ME.CCTL2.R = 0x00FE;
	/* Set Start address for core 1: Will reset and start */
#if defined(START_FROM_FLASH)
    MC_ME.CADDR2.R = 0x1080000 | 0x1;
#else
    MC_ME.CADDR2.R = 0x4006a800 | 0x1;
#endif /* defined(START_FROM_FLASH) */
	
#endif	
#if defined(TURN_ON_CPU2)
	/* enable core 2 in all modes */
	MC_ME.CCTL3.R = 0x00FE;
	/* Set Start address for core 2: Will reset and start */
#if defined(START_FROM_FLASH)
    MC_ME.CADDR3.R = 0x1100000 | 0x1;
#else
    MC_ME.CADDR3.R = 0x400d5000 | 0x1;
#endif /* defined(START_FROM_FLASH) */
	
#endif
#if defined(TURN_ON_CPU1) || defined(TURN_ON_CPU2)
	MC_ME.MCTL.R = (mctl & 0xffff0000ul) | KEY_VALUE1;
	MC_ME.MCTL.R =  mctl; /* key value 2 always from MCTL */
#endif
}

/*******************************************************************************
* Local functions
*******************************************************************************/


/*******************************************************************************
Function Name : HW_init
Engineer      : Martin Kovar
Date          : Dec-29-2015
Parameters    :
Modifies      :
Returns       :
Notes         : initialization of the hw for the purposes of this example
Issues        :
*******************************************************************************/
static void HW_Init(void)
{
	xcptn_xmpl ();              /* Configure and Enable Interrupts */
	SysClk_Init();
	InitPeriClkGen();
    GPIO_Init();
}

/*******************************************************************************
Function Name : SysClk_Init
Engineer      : David Chung
Date          : Aug-13-2018
Parameters    : NONE
Modifies      : NONE
Returns       : NONE
Notes         : Enable XOSC, PLL0, PLL1 and enter RUN0 with PLL1 as sys clk (200 MHz)
Issues        : NONE
*******************************************************************************/
// Enable XOSC, PLL0, PLL1 and enter RUN0 with PLL1 as sys clk (200 MHz)
void SysClk_Init(void)
{

	      CGM.AC3_SC.B.SELCTL = 1;		    //connect XOSC to the PLL0 input
	      CGM.AC4_SC.B.SELCTL = 1;		    //connect XOSC to the PLL1 input

	      // Set PLL0 to 50 MHz with 40MHz XOSC reference
	      PLLDIG.PLL0DV.R = 0x3008100A;	     // PREDIV =  1, MFD = 10, RFDPHI = 8, RFDPHI1 = 6

	      MC_ME.RUN_MC[0].R = 0x00130070;		    // RUN0 cfg: IRCON,OSC0ON,PLL0ON,syclk=IRC

	      // Mode Transition to enter RUN0 mode:
	      MC_ME.MCTL.R = 0x40005AF0;		    // Enter RUN0 Mode & Key
	      MC_ME.MCTL.R = 0x4000A50F;		    // Enter RUN0 Mode & Inverted Key
	      while (MC_ME.GS.B.S_MTRANS) {};		    // Wait for mode transition to complete
	      while(MC_ME.GS.B.S_CURRENT_MODE != 4) {};	    // Verify RUN0 is the current mode

	      // Set PLL1 to 200 MHz with 40MHz XOSC reference
	      PLLDIG.PLL1DV.R = 0x00020014;	     // MFD = 20, RFDPHI = 2

	      MC_ME.RUN_PC[0].R = 0x000000FE;		    // enable peripherals run in all modes
	      MC_ME.RUN_MC[0].R = 0x001300F4;		    // RUN0 cfg: IRCON, OSC0ON, PLL1ON, syclk=PLL1

	      CGM.SC_DC0.R = 0x80030000;    // SYS_CLK at syst clk div by 4 ... (50 MHz)

	      // Mode Transition to enter RUN0 mode:
	      MC_ME.MCTL.R = 0x40005AF0;		    // Enter RUN0 Mode & Key
	      MC_ME.MCTL.R = 0x4000A50F;		    // Enter RUN0 Mode & Inverted Key
	      while (MC_ME.GS.B.S_MTRANS) {};		    // Wait for mode transition to complete
	      while(MC_ME.GS.B.S_CURRENT_MODE != 4) {};	    // Verify RUN0 is the current mode

}


/*******************************************************************************
Function Name : PeriClkGen
Engineer      : David Chung
Date          : Aug-13-2018
Parameters    :
Modifies      :
Returns       :
Notes         : - Enable all auxiliary clocks, IMPORTANT - MOTC_CLK is set to 5MHz
Issues        :
*******************************************************************************/
void InitPeriClkGen(void)
{
	      CGM.AC0_SC.B.SELCTL = 0b010; //Select PLL0_PHI_CLK (50 MHz) as source of Aux Clk 0
	      CGM.AC0_DC0.R = 0x80090000;	//Divide by 10 (5 MHz) for MC_CLK
}


/*******************************************************************************
Function Name : GPIO_Init
Engineer      : David Chung
Date          : Aug-13-2018
Parameters    : NONE
Modifies      : NONE
Returns       : NONE
Notes         : SIUL2 initialization (ports),
Issues        : NONE
*******************************************************************************/
static void GPIO_Init(void)
{
   SIUL2.MSCR[LED1_pin].R = 0x32000000; 	//PA0 (P8.1) GPIO to be connected to LED
   SIUL2.GPDO[LED1_pin].B.PDO_4n = 1;
}


/*******************************************************************************
Function Name : ETimer_Init
Engineer      : David Chung
Date          : Aug-13-2018
Parameters    : NONE
Modifies      : NONE
Returns       : NONE
Notes         : ETimer1 channel 0, count mode for delay, Primary source is MC_CLK divided by 128
Issues        : NONE
*******************************************************************************/
static void ETimer_Init(void)
{
	    ETIMER_1.ENBL.R = 0x0;			// disable Timer0 channels
	    ETIMER_1.CH[0].CTRL1.R = 0x3F40;	 // Counts only rising edge of the MC_CLK (5MHz in RUN0), divide by 128, count up, count repeatedly, count until compare, then reinitialized
	    ETIMER_1.CH[0].COMP1.R = 0x9896;     // Compare value for 1 second delay (5MHz/128 = 39.063KHz)
	    ETIMER_1.CH[0].INTDMA.R = 0x0002;    // enable interrupt for COMP1
	    ETIMER_1.CH[0].CCCTRL.R = 0x0240;	 // compare on COMP1 when counting up, CAPT1 on rising edge,
	    ETIMER_1.ENBL.R = 0x0003;			// Enable Timer0 channel
}


/*******************************************************************************
Function Name : ETimer_ISR
Engineer      : David Chung
Date          : Aug-13-2018
Parameters    : NONE
Modifies      : NONE
Returns       : NONE
Notes         : ETimer interrupt routine
Issues        : NONE
*******************************************************************************/
void ETimer_ISR(void)
{
	ETIMER_1.CH[0].STS.R = 0x0002;                     //clear interrupt flag
	SIUL2.GPDO[LED1_pin].B.PDO_4n ^= 1;
}


int main(void)
{
	int counter = 0;

	HW_Init();
	INTC.PSR[622].R = 0x8001;    //set interrupt priority of eTimer1_ETC0

	ETimer_Init();

	    /* Loop forever */
	while (1)
	{
	    counter++;
	}

	return 0;
}


